locals {
  flowable_repo_username = var.flowable_repo_username != "" ? var.flowable_repo_username : jsondecode(data.aws_secretsmanager_secret_version.flowable_repo_auth.secret_string)["flowable_repo_username"]
  flowable_repo_password = var.flowable_repo_password != "" ? var.flowable_repo_password : jsondecode(data.aws_secretsmanager_secret_version.flowable_repo_auth.secret_string)["flowable_repo_password"]
  flowable_db_username   = var.flowable_db_username != "" ? var.flowable_db_username : jsondecode(data.aws_secretsmanager_secret_version.flowable_db_auth.secret_string)["flowable_db_username"]
  flowable_db_password   = var.flowable_db_password != "" ? var.flowable_db_password : jsondecode(data.aws_secretsmanager_secret_version.flowable_db_auth.secret_string)["flowable_db_password"]
  flowable_license       = data.aws_secretsmanager_secret_version.flowable_license.secret_string
}

data "aws_secretsmanager_secret_version" "flowable_repo_auth" {
  secret_id = var.aws_secret_flowable_repo_auth_id
}

data "aws_secretsmanager_secret_version" "flowable_license" {
  secret_id = var.aws_secret_flowable_license_id
}

data "aws_secretsmanager_secret_version" "flowable_db_auth" {
  secret_id = var.aws_secret_flowable_db_auth_id
}

data "aws_eks_cluster" "eks" {
  name = var.cluster_name
}

data "aws_eks_cluster_auth" "eks" {
  name = var.cluster_name
}

provider "kubernetes" {
  host                   = data.aws_eks_cluster.eks.endpoint
  cluster_ca_certificate = base64decode(data.aws_eks_cluster.eks.certificate_authority[0].data)
  token                  = data.aws_eks_cluster_auth.eks.token
}

provider "helm" {
  kubernetes {
    host                   = data.aws_eks_cluster.eks.endpoint
    cluster_ca_certificate = base64decode(data.aws_eks_cluster.eks.certificate_authority[0].data)
    token                  = data.aws_eks_cluster_auth.eks.token
  }
}


resource "kubernetes_namespace" "flowable" {
  metadata {
    name = var.namespace
  }
}

# Create a secret holding the Flowable docker registry authentication
resource "kubernetes_secret" "flowable_docker_auth" {
  metadata {
    name      = var.secret_name_docker_auth
    namespace = var.namespace
  }

  data = {
    ".dockerconfigjson" = jsonencode({
      auths = {
        "${var.flowable_docker_registry_server}" = {
          auth = "${base64encode("${local.flowable_repo_username}:${local.flowable_repo_password}")}"
        }
      }
    })
  }

  type = "kubernetes.io/dockerconfigjson"
}

# Create a secret holding the Flowable license
resource "kubernetes_secret" "flowable_license" {
  metadata {
    name      = var.secret_name_flowable_license
    namespace = var.namespace
  }

  data = {
    "flowable.license" = local.flowable_license
  }
}

# Create a secret holding the db credentials
resource "kubernetes_secret" "flowable_db_creds" {
  metadata {
    name      = "flowable-db-creds"
    namespace = var.namespace
  }

  data = {
    flowable-db-username = local.flowable_db_username,
    flowable-db-password = local.flowable_db_password
  }
}

# Create operator role for the Flowable namespace
resource "kubernetes_role" "flowable_operator_role" {
  metadata {
    name      = "flowable-admin"
    namespace = var.namespace
  }

  rule {
    api_groups = ["*"]
    resources  = ["*"]
    verbs      = ["*"]
  }
}

# Create dev role for the Flowable namespace
resource "kubernetes_role" "flowable_dev_role" {
  metadata {
    name      = "flowable-dev"
    namespace = var.namespace
  }

  rule {
    api_groups = [""]
    resources  = ["pods"]
    verbs      = ["get", "watch", "list"]
  }
}

resource "kubernetes_role_binding" "flowable_operator_role_binding" {
  metadata {
    name      = "flowable-admin_binding"
    namespace = var.namespace
  }
  role_ref {
    api_group = "rbac.authorization.k8s.io"
    kind      = "Role"
    name      = "flowable-admin"
  }

  subject {
    kind      = "Group"
    name      = "flowable-admin"
    api_group = "rbac.authorization.k8s.io"
  }
}

resource "kubernetes_role_binding" "flowable_dev_role_binding" {
  metadata {
    name      = "flowable-dev_binding"
    namespace = var.namespace
  }
  role_ref {
    api_group = "rbac.authorization.k8s.io"
    kind      = "Role"
    name      = "flowable-dev"
  }

  subject {
    kind      = "Group"
    name      = "flowable-dev"
    api_group = "rbac.authorization.k8s.io"
  }
}

# creates EFS dynamic provisioning storage class
resource "kubernetes_storage_class" "efs_sc_dyn" {
  metadata {
    name = "efs-sc-dyn-flowable"
  }
  storage_provisioner = "efs.csi.aws.com"
  reclaim_policy      = "Retain"
  parameters = {
    directoryPerms   = "700"
    fileSystemId     = var.efs_file_system_id
    provisioningMode = "efs-ap"
  }
}

module "efs_csi_driver" {
  source = "git::https://github.com/DNXLabs/terraform-aws-eks-efs-csi-driver.git"

  cluster_name                     = data.aws_eks_cluster.eks.id
  cluster_identity_oidc_issuer     = var.cluster_oidc_issuer_url
  cluster_identity_oidc_issuer_arn = var.cluster_oidc_issuer_arn
  create_storage_class             = false
}
