# Include the root terragrunt `root.hcl` configuration. The root configuration contains settings that are common across all
# components and environments, such as how to configure remote state.
include {
  path = find_in_parent_folders("root.hcl")
}

# We override the terraform block source attribute here just for the QA environment to show how you would deploy a
# different version of the module in a specific environment.
terraform {
  # source = "${local.base_source_url}"
  # source = "${local.base_source_url}?ref=v0.4.0"
  source = "../../../../../../flowable-iac-modules-aws//flowable-app"
}

locals {
  # Automatically load environment-level variables
  account_vars     = read_terragrunt_config(find_in_parent_folders("account.hcl"))
  environment_vars = read_terragrunt_config(find_in_parent_folders("env.hcl"))
  app_vars         = read_terragrunt_config(find_in_parent_folders("app.hcl"))

  # Extract out common variables for reuse
  workload                 = local.account_vars.locals.workload
  namespace                = local.environment_vars.locals.namespace
  env_suffix               = local.environment_vars.locals.env_suffix
  # external_domain          = local.app_vars.locals.external_domain
  es_enabled               = local.app_vars.locals.es_enabled ? true : false
  flowable_control_enabled = local.app_vars.locals.flowable_control_enabled ? true : false
  flowable_design_enabled  = local.app_vars.locals.flowable_design_enabled ? true : false
  flowable_work_enabled    = local.app_vars.locals.flowable_work_enabled ? true : false
  flowable_engage_enabled  = local.app_vars.locals.flowable_engage_enabled ? true : false

  # Expose the base source URL so different versions of the module can be deployed in different environments.
  base_source_url = "git@code.flowable.com:flw-lic/flowable-product/flowable-iac/flowable-iac-modules-aws.git//flowable-app"
}

dependencies {
  paths = ["../../foundation/k8s/k8s-config", "../../foundation/k8s/nginx-controller"]
}

dependency "k8s" {
  config_path = "../../foundation/k8s/eks-cluster"
}

dependency "database" {
  config_path = "../rds-postgres"
}

dependency "elasticsearch" {
  config_path  = "../elasticsearch-service"
  skip_outputs = local.es_enabled ? false : true
}

dependency "nginx_ingress" {
  config_path = "../../foundation/k8s/nginx-controller"
}

inputs = {
  cluster_name                     = dependency.k8s.outputs.cluster_name
  release_name                     = local.workload
  release_namespace                = local.namespace
  chart_name                       = "flowable"
  chart_version                    = "3.11.3"

  values = templatefile("values.yaml", {
    ingress_domain                      = dependency.nginx_ingress.outputs.load_balancer_hostname
    spring_elasticsearch_rest_uris      = (local.es_enabled ? "http://${dependency.elasticsearch.outputs.elk_endpoint}" : ""),
    flowable_indexing_index-name-prefix = "${local.workload}-",
    es_enabled                          = local.es_enabled
    flowable_control_enabled            = local.flowable_control_enabled
    flowable_design_enabled             = local.flowable_design_enabled
    flowable_work_enabled               = local.flowable_work_enabled
    flowable_engage_enabled             = local.flowable_engage_enabled
    spring_datasource_driver-class-name = "org.postgresql.Driver",
    spring_datasource_url_control       = (local.flowable_control_enabled ? "jdbc:postgresql://${dependency.database.outputs.db_control_endpoint}/${dependency.database.outputs.db_control_name}" : ""),
    spring_datasource_url_design        = (local.flowable_design_enabled ? "jdbc:postgresql://${dependency.database.outputs.db_design_endpoint}/${dependency.database.outputs.db_design_name}" : ""),
    spring_datasource_url_work          = (local.flowable_work_enabled ? "jdbc:postgresql://${dependency.database.outputs.db_work_endpoint}/${dependency.database.outputs.db_work_name}" : ""),
    spring_datasource_url_engage        = (local.flowable_engage_enabled ? "jdbc:postgresql://${dependency.database.outputs.db_engage_endpoint}/${dependency.database.outputs.db_engage_name}" : "")
  })
}
  