# ---------------------------------------------------------------------------------------------------------------------
# TERRAGRUNT CONFIGURATION
# Terragrunt is a thin wrapper for Terraform that provides extra tools for working with multiple Terraform modules,
# remote state, and locking: https://github.com/gruntwork-io/terragrunt
# ---------------------------------------------------------------------------------------------------------------------

locals {
  # merge all configuration properties into a single map
  # mind! properties can be overwritten
  common_vars = merge(
    read_terragrunt_config(find_in_parent_folders("account.hcl", "non_existing.hcl"), {locals = {}}).locals,
    read_terragrunt_config(find_in_parent_folders("region.hcl", "non_existing.hcl"), {locals = {}}).locals,
    read_terragrunt_config(find_in_parent_folders("env.hcl", "non_existing.hcl"), {locals = {}}).locals,
    read_terragrunt_config(find_in_parent_folders("foundation.hcl", "non_existing.hcl"), {locals = {}}).locals,
    read_terragrunt_config(find_in_parent_folders("app.hcl", "non_existing.hcl"), {locals = {}}).locals
  )

  flowable_iac_tag                  = local.common_vars.env_suffix
  full_workload                     = "${local.common_vars.workload}-${local.common_vars.env_suffix}"

  dev_ops_admin_role_arn            = "arn:aws:iam::${local.common_vars.account_id}:role/flowable/DevOpsAdminRole"
  flowable_admin_role_arn           = "arn:aws:iam::${local.common_vars.account_id}:role/flowable/FlowableAdminRole"
  flowable_dev_role_arn             = "arn:aws:iam::${local.common_vars.account_id}:role/flowable/FlowableDevRole"

  # The role ARN that needs access the Terraform state resources
  tf_remote_state_role_arn = "arn:aws:iam::${local.common_vars.sec_account_id}:role/terraform/TerraformBackend"
}

# Set the AWS profile
terraform {
  extra_arguments "aws_profile" {
    commands = [
      "init",
      "apply",
      "refresh",
      "import",
      "plan",
      "taint",
      "untaint"
    ]

    env_vars = {
      AWS_PROFILE = "${local.common_vars.aws_profile}"
    }
  }
}

# Generate an AWS provider block
generate "provider" {
  path      = "provider.tf"
  if_exists = "overwrite_terragrunt"
  contents  = <<EOF
  provider "aws" {
    region = "${local.common_vars.region}"
    default_tags {
      tags = {
        flowable-iac = "${local.flowable_iac_tag}"
      }
    }
    assume_role {
      role_arn    = "${local.common_vars.tf_assume_role_arn}"
    }
    profile = "${local.common_vars.aws_profile}"
  }
  EOF
}

# Configure Terragrunt to automatically store tfstate files in an S3 bucket
remote_state {
  backend = "s3"
  config = {
    bucket         = "flw-iac-tf-state"
    key            = "${path_relative_to_include()}/terraform.tfstate"
    region         = "${local.common_vars.region}"
    encrypt        = true
    key            = "${path_relative_to_include()}/terraform.tfstate"
    role_arn       = "${local.tf_remote_state_role_arn}"

    dynamodb_table            = "flw-iac-tf-state-locks"
    accesslogging_bucket_name = "flw-iac-tf-state-logs"

    profile = "${local.common_vars.aws_profile}"
  }
  generate = {
    path      = "backend.tf"
    if_exists = "overwrite_terragrunt"
  }
}

# ---------------------------------------------------------------------------------------------------------------------
# GLOBAL PARAMETERS
# These variables apply to all configurations in this subfolder. These are automatically merged into the child
# `terragrunt.hcl` config via the include block.
# ---------------------------------------------------------------------------------------------------------------------

# Configure root level variables that all resources can inherit. This is especially helpful with multi-account configs
# where terraform_remote_state data sources are placed directly into the modules.
inputs = merge(
  {
    full_workload: local.full_workload
  },
  {
    flowable_iac_tag: local.flowable_iac_tag
  },
  local.common_vars
)